MODULE MsgLoop;
(* PURPOSE: Demonstrate basic Windows program  Message Processing *)
(*                                                                            *)
(* FUNCTIONS:                                                                 *)
(*   MainWndProc() - processes messages                                       *)
(*                                                                            *)
(* COMMENTS:                                                                  *)
(*   Windows can have several copies of your application running at the       *)
(*   same time.  The variable hInst keeps track of which instance this        *)
(*   application is so that processing will be to the correct window.         *)
(******************************************************************************)

IMPORT
   W := Windows, S := SYSTEM, ML := MsgLog, M := myRtns;

CONST
  class = "helloWclass";
  titleBar = "Message Loop Demo";
  msg = "Hello Windows";

VAR
  hInst:    W.HANDLE;
  hScroll:  W.HWND;
  sPos:     INTEGER;
  
(******************************************************************************)
(* FUNCTION: MainWndProc(HWND, unsigned, WORD, LONG)                          *)
(*                                                                            *)
(* PURPOSE:  Processes messages                                               *)
(*                                                                            *)
(* MESSAGES:                                                                  *)
(*   W.WM_COMMAND    - application menu (About dialog box)                    *)
(*   W.WM_DESTROY    - destroy window                                         *)
(*                                                                            *)
(******************************************************************************)

PROCEDURE [WINDOWS] MainWndProc*(hWnd: W.HWND;   (* window handle             *)
                                 message: W.WORD;(* type of message           *)
                                 wParam: W.WORD; (* additional information    *)
                                 lParam: LONGINT (* additional information    *)
                                ): LONGINT;
                                
CONST
  button = "button";
  Scroll = "scrollbar";  
  PB = "PushButton";
  CB = "CheckBox";                              

  VAR
    rc: INTEGER;
    hdc:  W.HDC;
    ps:   W.PAINTSTRUCT;
    rect: W.RECT;
    hPB, hCB, 
    hRB:    W.HWND;
    
BEGIN
  CASE message OF
    W.WM_CREATE:      (*   Create some buttons on the Window *)
      hPB := W.CreateWindow(S.ADR(button), S.ADR(PB),
             W.WS_CHILD + W.WS_VISIBLE + W.BS_PUSHBUTTON,
             10, 10, 100, 25,
             hWnd, 101, hInst, W.NULL);
      hPB := W.CreateWindow(S.ADR(button), S.ADR(CB),
             W.WS_CHILD + W.WS_VISIBLE + W.BS_AUTOCHECKBOX,
             10, 40, 100, 25,
             hWnd, 102, hInst, W.NULL);
      hScroll := W.CreateWindow(
             S.ADR(Scroll), W.NULL, 
             W.WS_CHILD + W.WS_VISIBLE + W.SBS_VERT, 
             150, 0, 15, 200,
             hWnd, W.NULL, 
             hInst, W.NULL);
      sPos := 50;       
      W.SetScrollRange(hScroll, W.SB_CTL, 0, 100, W.false);
      rc := W.SetScrollPos(hScroll, W.SB_CTL, sPos, W.false);
      RETURN 0;       
  | W.WM_PAINT:
      hdc := W.BeginPaint(hWnd, S.ADR(ps));
      W.GetClientRect(hWnd, S.ADR(rect));
      rc := W.DrawText(hdc, S.ADR(msg), -1, S.ADR(rect),
              W.DT_SINGLELINE + W.DT_CENTER + W.DT_VCENTER); 
      W.EndPaint(hWnd, S.ADR(ps)); 
      RETURN 0;
  | W.WM_MOUSEMOVE:
      ML.LogMsgSI("WM_MOUSEMOVE: Mx=", M.LoWord(lParam), FALSE);  
      ML.LogMsgSI(", My=", M.HiWord(lParam), TRUE);
      RETURN 0;
  | W.WM_LBUTTONDOWN:    
      ML.LogMsgSI("WM_LBUTTONDOWN: Mx=", M.LoWord(lParam), FALSE);  
      ML.LogMsgSI(", My=", M.HiWord(lParam), TRUE);
      RETURN 0;
  | W.WM_COMMAND:
      IF wParam = 101 THEN
        ML.LogMsg("Push Button Clicked.", TRUE);
        RETURN 0;
      ELSIF wParam = 102 THEN
        ML.LogMsg("CheckBox Clicked.", TRUE);
        RETURN 0;  
      ELSE
      ML.LogMsgSI("Msg=", message, FALSE);
      ML.LogMsgSI(" Mx=", M.LoWord(lParam), FALSE);  
      ML.LogMsgSI(", My=", M.HiWord(lParam), TRUE);
      END;      
  | W.WM_DESTROY: 
    W.PostQuitMessage(0);
    RETURN 0;
  | W.WM_VSCROLL:  
      ML.LogMsgSI("VScroll=", message, FALSE);
      CASE wParam OF
        W.SB_PAGEDOWN: sPos := sPos + 10; ML.LogMsg(" wP=SB_PAGEDOWN ", FALSE);
      | W.SB_LINEDOWN: sPos := sPos + 1;  ML.LogMsg(" wP=SB_LINEDOWN ", FALSE);
      | W.SB_PAGEUP:   sPos := sPos - 10; ML.LogMsg(" wP=SB_PAGEUP ", FALSE);
      | W.SB_LINEUP:   sPos := sPos - 1;  ML.LogMsg(" wP=SB_LINEUP ", FALSE);
      | W.SB_THUMBTRACK, W.SB_THUMBPOSITION:
          sPos := M.LoWord(lParam); ML.LogMsg(" wP=SB_THUMBTRACK ", FALSE);
      | W.SB_ENDSCROLL:  ML.LogMsg(" wP=SB_ENDSCROLL ", FALSE);    
      ELSE  
      END;
      IF sPos < 0 THEN sPos := 0 END;
      IF sPos > 100 THEN sPos := 100 END;
      rc := W.SetScrollPos(hScroll, W.SB_CTL, sPos, W.true);
        ML.LogMsgSI(" Mx=", M.LoWord(lParam), FALSE);  
        ML.LogMsgSI(", My=", M.HiWord(lParam), TRUE);
      RETURN 0;
  ELSE                                    (* Passes it on if unproccessed     *)
    RETURN W.DefWindowProc(hWnd, message, wParam, lParam)
  END;  
END MainWndProc;

(******************************************************************************)
(* FUNCTION: WinMain(HANDLE, HANDLE, LPSTR, int)                              *)
(*                                                                            *)
(* PURPOSE: calls initialization function, processes message loop             *)
(*                                                                            *)
(* COMMENTS:                                                                  *)
(*   Windows recognizes this function by name as the initial entry point      *)
(*   for the program.  This function calls the application initialization     *)
(*   routine, if no other instance of the program is running, and always      *)
(*   calls the instance initialization routine.  It then executes a message   *)
(*   retrieval and dispatch loop that is the top-level control structure      *)
(*   for the remainder of execution.  The loop is terminated when a WM_QUIT   *)
(*   message is received, at which time this function exits the application   *)
(*   instance by returning the value passed by PostQuitMessage().             *)
(*                                                                            *)
(*   If this function must abort before entering the message loop, it         *)
(*   returns the conventional value W.NULL.                                   *)
(******************************************************************************)

PROCEDURE [WINDOWS] WinMain*(hInstance: W.HANDLE;    (* current instance      *)
                             hPrevInstance: W.HANDLE;(* previous instance     *)
                             lpCmdLine: W.LPSTR;     (* command line          *)
                             nCmdShow: INTEGER       (* show-window type      *)
                            ): INTEGER;              (*   (open/icon)         *)
  VAR r: LONGINT; msg: W.MSG;             (* message                          *)
    rc:  INTEGER;
    wc: W.WNDCLASS;
    hWnd: W.HWND;                           (* Main window handle               *)
BEGIN
  IF hPrevInstance = 0 THEN               (* Other instances of app running?  *)
(* COMMENTS:                                                                  *)
(*   If no othere instances of the program are running - we perform           *)
(*   and initialization tasks that can be done for any number of              *)
(*   running instances.                                                       *)
(*                                                                            *)
(*   In this case, we initialize a window class by filling out a data         *)
(*   structure of type WNDCLASS and calling the Windows RegisterClass()       *)
(*   function.  Since all instances of this application use the same window   *)
(*   class, we only need to do this when the first instance is initialized.   *)
                                             (* Fill in window class structure   *)
                                            (* with parameters that describe    *)
                                            (* the main window.                 *)
    wc.style := W.NULL;                     (* Class style(s).                  *)
    wc.lpfnWndProc := MainWndProc;          (* Function to retrieve messages for*)
                                            (* windows of this class.           *)
    wc.cbClsExtra := 0;                     (* No per-class extra data.         *)
    wc.cbWndExtra := 0;                     (* No per-window extra data.        *)
    wc.hInstance := hInstance;              (* Application that owns the class. *)
    wc.hIcon := W.LoadIcon(hInstance, S.ADR("GENERICICON"));
    wc.hCursor := W.LoadCursor(W.NULL, W.IDC_ARROW);
    wc.hbrBackground := W.GetStockObject(W.WHITE_BRUSH);
    wc.lpszMenuName := W.NULL;              (* Name of menu resource in .RC file*)
    wc.lpszClassName := S.ADR(class);  (* Name used in call to CreateWindow*)
                                          (* Register the window class and    *)
                                          (*  return success/failure code.    *)
    rc := W.RegisterClass(S.ADR(wc));
    
    ML.DefineMsgLogClass(hInstance);      (* Default Window registered in MsgLog Module *)
  END;
 

                                          (* Save the instance handle in      *)
                                          (* static variable, which will be   *)
  hInst := hInstance;                     (* used in many subsequent calls    *)
                                          
                                          
(* Perform initializations that apply to a specific instance     *)

  r := ML.DisplayMsgLog(hInst, nCmdShow); (* Displays the Message Log Window *)
  
                                          (* Create a main window for this    *)
                                          (*  application instance            *)
  hWnd := W.CreateWindow(S.ADR(class),(* See RegisterClass() call        *)
                         S.ADR(titleBar),(* Text for window title bar    *)
                         W.WS_OVERLAPPEDWINDOW, (* Window style               *)
                         300, (* Default horizontal position      *)
                         0, (* Default vertical position        *)
                         W.CW_USEDEFAULT, (* Default width                    *)
                         W.CW_USEDEFAULT, (* Default height                   *)
                         W.NULL,          (* Overlapped windows have no parent*)
                         W.NULL,          (* Use the window class menu        *)
                         hInstance,       (* This instance owns this window   *)
                         W.NULL);         (* Pointer not needed               *)
                                          (* If window could not be created,  *)
  IF hWnd = 0 THEN RETURN W.false END;      (* return "failure"                 *)
                                          (* Make the window visible; update  *)
                                          (* its client area; and return      *)
                                          (* "success"                        *)
  r := W.ShowWindow(hWnd, nCmdShow);      (* Show the window                  *)
  W.UpdateWindow(hWnd);                   (* Sends WM_PAINT message           *)
  
  
                                          (* Acquire and dispatch messages    *)
                                          (* until a WM_QUIT message is       *)
                                          (* received.                        *)
  WHILE W.GetMessage(S.ADR(msg),     (* message structure                *)
                     W.NULL,              (* handle of window receiving the ms*)
                     W.NULL,              (* lowest message to examine        *)
                     W.NULL) # 0 DO       (* highest message to examine       *)

    r := W.TranslateMessage(S.ADR(msg)); (* Translates virtual key codes *)
    r := W.DispatchMessage(S.ADR(msg))   (* Dispatches message to window *)
  END;
  RETURN msg.wParam
END WinMain;

BEGIN
END MsgLoop.
